/* Copyright (C) 2014-2018 RealVNC Ltd. All Rights Reserved.
 *
 * This is sample code intended to demonstrate part of the
 * VNC Mobile Solution SDK. It is not intended as a production-ready
 * component.
 */

#include "StringParser.h"
#include <DiscoveryException.h>
#include <assert.h>

using namespace vncdiscovery;

namespace
{
  VNCDiscoverySDKParser* createParser(VNCDiscoverySDKUtils *pUtils,
      const char* string)
  {
    if(!pUtils)
    {
      throw DiscoveryException(
          "Unable to create Parser, Utils structure is NULL",
          VNCDiscoverySDKErrorInvalidParameter);
    }
    if(!pUtils->createParser)
    {
      throw DiscoveryException(
          "Unable to create Parser, the operation is not supported",
          VNCDiscoverySDKErrorNotSupported);
    }

    VNCDiscoverySDKParser* pParser = NULL;
    VNCDiscoverySDKError err = pUtils->createParser(
        pUtils->pSDKInstance, string, &pParser);

    if(err != VNCDiscoverySDKErrorNone)
    {
      assert(!pParser);
      if(pParser && pUtils->parserDestroy)
      {
        pUtils->parserDestroy(pParser);
      }
      throw DiscoveryException(
          "Error when trying to create Parser", err);
    }
    else if(!pParser)
    {
      // Shouldn't really get into here. If the SDK returned error none, then
      // the Parser should have been created and initialized
      assert(false);
      throw DiscoveryException("Parser was not created",
          VNCDiscoverySDKErrorNotInitialized);
    }
    return pParser;
  }
}

StringParser::StringParser(VNCDiscoverySDKUtils *pUtils, const char* string)
  : mpUtils(pUtils),
    mpParser(NULL)
{
  mpParser = createParser(pUtils, string);
  assert(mpParser);
}

StringParser::~StringParser()
{
  if(mpParser && mpUtils->parserDestroy)
  {
    mpUtils->parserDestroy(mpParser);
  }
}

const char* StringParser::getStringField(const char* fieldPath,
    VNCDiscoverySDKError* pError)
{
  const char* pResult = NULL;
  VNCDiscoverySDKError err = VNCDiscoverySDKErrorNotSupported;
  if(mpUtils->parserGetStringField)
  {
    err = mpUtils->parserGetStringField(mpParser, fieldPath, &pResult);
  }
  if(pError)
  {
    *pError = err;
  }
  return pResult;
}

vnc_int32_t StringParser::getInt32Field(const char* fieldPath,
    VNCDiscoverySDKError* pError)
{
  vnc_int32_t result = VNC_DISCOVERY_SDK_UTILS_INT_FAIL_VALUE;
  VNCDiscoverySDKError err = VNCDiscoverySDKErrorNotSupported;
  if(mpUtils->parserGetInt32Field)
  {
    err = mpUtils->parserGetInt32Field(mpParser, fieldPath, &result);
  }
  if(pError)
  {
    *pError = err;
  }
  return result;
}

vnc_int32_t StringParser::getSubFieldCount(const char* fieldPath)
{
  if(!mpUtils->parserGetSubFieldCount)
  {
    return -1;
  }
  return mpUtils->parserGetSubFieldCount(mpParser, fieldPath);
}

